<?php
/*
 * Plugin Name:       User Registration Using Elementor Form
 * Plugin URI:        https://ahmedmoustafa.me/plugins/user-registration-using-elementor-form
 * Description:       Stop using rigid default registration pages. Build beautiful, custom registration flows using the Elementor drag-and-drop builder—no coding required.
 * Version:           1.1
 * Requires at least: 5.2
 * Requires PHP:      7.2
 * Author:            Ahmed Mostafa
 * Author URI:        https://ahmedmoustafa.me
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       user-registration-using-elementor-form
 * Domain Path:       /languages
 * Requires Plugins:  elementor
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

final class User_Registration_Using_Elementor_Form {

	const VERSION = '1.1';
	const ELEMENTOR_PRO_MIN_VERSION = '2.0.0';

	private static $_instance = null;

	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'on_plugins_loaded' ] );
		// Use a more reliable filter to inject mapping data directly into the HTML
		add_filter( 'elementor/widget/render_content', [ $this, 'inject_form_mapping_data' ], 10, 2 );
	}

	public function on_plugins_loaded() {
		// Include and register the custom action functionality
		add_action( 'elementor_pro/forms/actions/register', [ $this, 'register_form_actions' ] );

		// Check if Elementor Pro is active for admin notice
		if ( ! defined( 'ELEMENTOR_PRO_VERSION' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_elementor_pro' ] );
		}
	}

	public function admin_notice_missing_elementor_pro() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verification not required for displaying/clearing a transient-like notice.
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
			/* translators: 1: Plugin Name 2: Elementor Pro */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'user-registration-using-elementor-form' ),
			'<strong>' . esc_html__( 'User Registration Using Elementor Form', 'user-registration-using-elementor-form' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor Pro', 'user-registration-using-elementor-form' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', wp_kses_post( $message ) );
	}

	public function register_form_actions( $form_actions_registrar ) {
		require_once( __DIR__ . '/includes/class-action-user-registration.php' );

		$form_actions_registrar->register( new \User_Registration_Using_Elementor_Form\Action_User_Registration() );
	}

	public function init_login_handler() {
		require_once( __DIR__ . '/includes/class-login-handler.php' );
		new \User_Registration_Using_Elementor_Form\Login_Handler();
	}
    
    public function init_rest_api() {
        require_once( __DIR__ . '/includes/class-rest-api.php' );
        new \User_Registration_Using_Elementor_Form\Rest_Api();
    }
    
    public function enqueue_scripts() {
        wp_enqueue_script( 
            'am-user-reg-validation', 
            plugin_dir_url( __FILE__ ) . 'assets/js/validation.js', 
            [ 'jquery' ], 
            self::VERSION, 
            true 
        );
        
        wp_localize_script( 'am-user-reg-validation', 'amUserReg', [
            'restUrl' => rest_url( 'am-user-reg/v1/' ),
        ] );
    }
    
    public function init_verification() {
        require_once( __DIR__ . '/includes/class-verification-handler.php' );
        new \User_Registration_Using_Elementor_Form\Verification_Handler();
    }

    public function inject_form_mapping_data( $content, $widget ) {
		if ( 'form' !== $widget->get_name() ) {
			return $content;
		}

		$settings = $widget->get_settings_for_display();
		$actions = isset( $settings['submit_actions'] ) ? (array) $settings['submit_actions'] : [];
		
		if ( ! in_array( 'user_registration', $actions, true ) ) {
			return $content;
		}

		$map = [
			'username'         => isset( $settings['field_map_username'] ) ? $settings['field_map_username'] : '',
			'email'            => isset( $settings['field_map_email'] ) ? $settings['field_map_email'] : '',
			'password'         => isset( $settings['field_map_password'] ) ? $settings['field_map_password'] : '',
			'confirm_password' => isset( $settings['field_map_confirm_password'] ) ? $settings['field_map_confirm_password'] : '',
			'billing_phone'    => isset( $settings['field_map_billing_phone'] ) ? $settings['field_map_billing_phone'] : '',
		];

		$json_map = esc_attr( json_encode( $map ) );
		
		// Method 1: Inject into the widget wrapper (Elementor way)
		$widget->add_render_attribute( '_wrapper', 'data-am-user-reg-map', $json_map );

		// Method 2: String replacement for the <form> tag (Direct way)
		if ( strpos( $content, '<form' ) !== false ) {
			$content = str_replace( '<form ', '<form data-am-user-reg-map="' . $json_map . '" ', $content );
		}

		return $content;
	}
}

$plugin = User_Registration_Using_Elementor_Form::instance();
$plugin->init_login_handler();
$plugin->init_rest_api();
$plugin->init_verification();
add_action( 'wp_enqueue_scripts', [ $plugin, 'enqueue_scripts' ] );