<?php
namespace User_Registration_Using_Elementor_Form;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Login_Handler {

	public function __construct() {
		add_filter( 'authenticate', [ $this, 'authenticate_with_phone' ], 5, 3 );
		add_filter( 'authenticate', [ $this, 'check_verification_status' ], 21, 3 );
		add_action( 'wp_login_failed', [ $this, 'handle_login_failure' ] );
	}
	
	public function handle_login_failure( $username ) {
	    // We need to check if the error was 'not_verified'.
	    // wp_login_failed passes the username. We might need to check the global $error if accessible or re-authenticate loosely?
	    // Actually, wp_authenticate executes before wp_login_failed. If wp_authenticate returns error, wp_login_failed fires.
	    // But wp_login_failed doesn't receive the error object directly in all versions? 
	    // Wait, the 'wp_login_failed' action is fired *after* authentication fails.
	    
	    // Better approach: 'authenticate' filter returns WP_Error. WordPress login handling usually redirects to wp-login.php with error.
	    // To intercept this redirection, we might need to hook into 'login_redirect' but that's for success.
	    // Or strictly check 'authenticate' and if we return error, we can handle redirect there? 
	    // NO, 'authenticate' filters should return error object. 
	    
	    // If we want to redirect *instead* of standard WP behavior for this specific error:
	    // We can do it inside the authenticate filter itself, BUT that breaks other plugins effectively.
	    
	    // Security: Verify nonce for login form processing
	    if ( ! empty( $_POST ) && isset( $_POST['log'] ) ) {
	        check_admin_referer( 'log-in' );
	    }

	    $user = get_user_by( 'login', $username );
	    if ( ! $user ) {
	        $user = get_user_by( 'email', $username );
	    }
	    
	    if ( $user ) {
	        $is_verified = get_user_meta( $user->ID, 'am_is_verified', true );
	        if ( '0' === $is_verified ) {
	             // Use Referer
	             if ( isset( $_SERVER['HTTP_REFERER'] ) ) {
	                 $referer = esc_url_raw( wp_unslash( $_SERVER['HTTP_REFERER'] ) );
	                 // Avoid redirect loops if referer is wp-login
	                 if ( strpos( $referer, 'wp-login.php' ) === false ) {
	                     wp_safe_redirect( add_query_arg( 'am_error', 'not_verified', $referer ) );
	                     exit;
	                 }
	             }
	        }
	    }
	}

	public function authenticate_with_phone( $user, $username, $password ) {
		// If user is already found or no username provided, return.
		if ( $user instanceof \WP_User || empty( $username ) ) {
			return $user;
		}

		// Try to find user by billing_phone meta
		$user_query = new \WP_User_Query( [
			'meta_key'    => 'billing_phone', // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
			'meta_value'  => $username, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
			'number'      => 1,
			'fields'      => 'ID',
		] );

		$user_ids = $user_query->get_results();

		if ( ! empty( $user_ids ) ) {
			$found_user = get_user_by( 'id', $user_ids[0] );
			
			if ( wp_check_password( $password, $found_user->user_pass, $found_user->ID ) ) {
				// Check verification status
				$is_verified = get_user_meta( $found_user->ID, 'am_is_verified', true );
				if ( '0' === $is_verified ) {
					return new \WP_Error( 'not_verified', esc_html__( 'Please verify your email before logging in.', 'user-registration-using-elementor-form' ) );
				}
				return $found_user;
			} else {
				// If phone matched but password failed, return error immediately to stop fallthrough
				return new \WP_Error( 'authentication_failed', __( '<strong>Error</strong>: The password you entered for the phone number is incorrect.', 'user-registration-using-elementor-form' ) );
			}
		}

		return $user;
	}
	
	// Add a general filter to check verification for normal logins (email/username)
	public function check_verification_status( $user, $username, $password ) {
	    if ( $user instanceof \WP_User ) {
	        $is_verified = get_user_meta( $user->ID, 'am_is_verified', true );
			if ( '0' === $is_verified ) {
				return new \WP_Error( 'not_verified', esc_html__( 'Please verify your email before logging in.', 'user-registration-using-elementor-form' ) );
			}
	    }
	    return $user;
	}
}

