<?php
namespace User_Registration_Using_Elementor_Form;

use Elementor\Controls_Manager;
use ElementorPro\Modules\Forms\Classes\Action_Base;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Action_User_Registration extends Action_Base {

	public function get_name() {
		return 'user_registration';
	}

	public function get_label() {
		return esc_html__( 'User Registration', 'user-registration-using-elementor-form' );
	}

	public function register_settings_section( $widget ) {
		$widget->start_controls_section(
			'section_user_registration',
			[
				'label' => esc_html__( 'User Registration', 'user-registration-using-elementor-form' ),
				'condition' => [
					'submit_actions' => $this->get_name(),
				],
			]
		);


        
        // Note: Field mapping in Elementor Forms usually requires a specific structure or usage of repeater values if dynamic.
        // However, standard Action_Base implementation usually just asks for the field ID or uses a select that lists form fields.
        // Elementor Pro's standard actions often use a loop to populate these options based on the form fields.
        // Since we are in the backend, we might not have easy access to the field names dynamically without JS.
        // But typically, Elementor actions allow manual entry of the Field ID (shortcode).
        // Let's look at how other actions do it. Usually they provide a select list of fields. 
        // For simplicity and robustness, we can let user type the Field ID or try to populate it if possible.
        // Elementor's 'mail' action uses 'mapping' controls? No, it uses standard controls.
        // Let's stick to simple text inputs for Field IDs first, or check if we can replicate the field selection behavior.
        // Actually, many add-ons just ask for "Field Shortcode/ID" e.g. [field_id].
        
        // Clarification: Elementor's native actions (like Mailchimp) have a field mapping repeater or individual selects.
        // The individual selects are populated via JS.
        // For this implementation, I will just use standard text controls where the user inputs the Field ID (e.g. 'name', 'email').
        // This is safer and easier to implement without complex JS injection.

		$widget->add_control(
			'field_map_username',
			[
				'label' => esc_html__( 'Username Field ID', 'user-registration-using-elementor-form' ),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__( 'Enter the Field ID for the Username (e.g. username).', 'user-registration-using-elementor-form' ),
			]
		);

		$widget->add_control(
			'field_map_email',
			[
				'label' => esc_html__( 'Email Field ID', 'user-registration-using-elementor-form' ),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__( 'Enter the Field ID for the Email (e.g. email).', 'user-registration-using-elementor-form' ),
			]
		);

		$widget->add_control(
			'field_map_password',
			[
				'label' => esc_html__( 'Password Field ID', 'user-registration-using-elementor-form' ),
				'type' => Controls_Manager::TEXT,
			]
		);

		$widget->add_control(
			'field_map_confirm_password',
			[
				'label' => esc_html__( 'Confirm Password Field ID', 'user-registration-using-elementor-form' ),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__( 'Enter the Field ID for the Confirm Password field.', 'user-registration-using-elementor-form' ),
			]
		);

		$widget->add_control(
			'field_map_first_name',
			[
				'label' => esc_html__( 'First Name Field ID', 'user-registration-using-elementor-form' ),
				'type' => Controls_Manager::TEXT,
			]
		);

		$widget->add_control(
			'field_map_last_name',
			[
				'label' => esc_html__( 'Last Name Field ID', 'user-registration-using-elementor-form' ),
				'type' => Controls_Manager::TEXT,
			]
		);

		$widget->add_control(
			'user_role',
			[
				'label' => esc_html__( 'User Role', 'user-registration-using-elementor-form' ),
				'type' => Controls_Manager::SELECT,
				'options' => $this->get_user_roles(),
				'default' => 'customer',
			]
		);
		
		$widget->add_control(
			'heading_woo_billing',
			[
				'label' => esc_html__( 'WooCommerce Billing', 'user-registration-using-elementor-form' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$widget->add_control(
			'field_map_billing_phone',
			[
				'label' => esc_html__( 'Billing Phone Field ID', 'user-registration-using-elementor-form' ),
				'type' => Controls_Manager::TEXT,
			]
		);
		
        // Add more billing fields as needed...

		$widget->add_control(
			'redirect_url',
			[
				'label' => esc_html__( 'Login Page URL', 'user-registration-using-elementor-form' ),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__( 'Enter the URL of the login page to redirect to after verification.', 'user-registration-using-elementor-form' ),
			]
		);
		
		$widget->end_controls_section();
	}

	public function run( $record, $ajax_handler ) {
		$settings = $record->get( 'form_settings' );
		$raw_fields = $record->get( 'fields' );
		
		// Normalize fields for easier access
		$fields = [];
		foreach ( $raw_fields as $id => $field ) {
			$fields[ $id ] = $field['value'];
		}

		$username_field = isset( $settings['field_map_username'] ) ? $settings['field_map_username'] : '';
		$email_field = isset( $settings['field_map_email'] ) ? $settings['field_map_email'] : '';
		$password_field = isset( $settings['field_map_password'] ) ? $settings['field_map_password'] : '';
		$confirm_password_field = isset( $settings['field_map_confirm_password'] ) ? $settings['field_map_confirm_password'] : '';
		
		$username = isset( $fields[ $username_field ] ) ? sanitize_user( $fields[ $username_field ] ) : '';
		$email = isset( $fields[ $email_field ] ) ? sanitize_email( $fields[ $email_field ] ) : '';
		$password = isset( $fields[ $password_field ] ) ? $fields[ $password_field ] : wp_generate_password();
		
		if ( empty( $username ) || empty( $email ) ) {
			$ajax_handler->add_error_message( esc_html__( 'Username and Email are required.', 'user-registration-using-elementor-form' ) );
			return;
		}

		// Validation: Confirm Password
		if ( ! empty( $confirm_password_field ) && isset( $fields[ $confirm_password_field ] ) ) {
			if ( $password !== $fields[ $confirm_password_field ] ) {
				$ajax_handler->add_error( $confirm_password_field, esc_html__( 'Passwords do not match.', 'user-registration-using-elementor-form' ) );
				return;
			}
		}

		// Validation: Password Requirements
		if ( ! empty( $password_field ) ) {
			// At least 8 characters, one uppercase, one special character
			if ( strlen( $password ) < 8 || 
				 ! preg_match( '/[A-Z]/', $password ) || 
				 ! preg_match( '/[^A-Za-z0-9]/', $password ) 
			) {
				$ajax_handler->add_error( $password_field, esc_html__( 'Password must be at least 8 characters long and include at least one uppercase letter and one special character.', 'user-registration-using-elementor-form' ) );
				return;
			}
		}

		if ( username_exists( $username ) ) {
			$ajax_handler->add_error( $username_field, esc_html__( 'This username is already taken.', 'user-registration-using-elementor-form' ) );
			return;
		}

		if ( email_exists( $email ) ) {
			$ajax_handler->add_error( $email_field, esc_html__( 'This email is already registered.', 'user-registration-using-elementor-form' ) );
			return;
		}

		// Validation: Billing Phone Uniqueness
		$billing_phone_field = isset( $settings['field_map_billing_phone'] ) ? $settings['field_map_billing_phone'] : '';
		if ( ! empty( $billing_phone_field ) && isset( $fields[ $billing_phone_field ] ) ) {
			$phone = $fields[ $billing_phone_field ];
			$user_query = new \WP_User_Query( [
				'meta_key'     => 'billing_phone', // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
				'meta_value'   => $fields[ $billing_phone_field ], // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
				'number'       => 1,
				'fields'       => 'ID',
			] );
			if ( ! empty( $user_query->get_results() ) ) {
				$ajax_handler->add_error( $billing_phone_field, esc_html__( 'This phone number is already registered.', 'user-registration-using-elementor-form' ) );
				return;
			}
		}
		
		$user_data = [
			'user_login' => $username,
			'user_email' => $email,
			'user_pass'  => $password,
			'role'       => isset( $settings['user_role'] ) ? $settings['user_role'] : 'customer',
		];
		
		if ( ! empty( $settings['field_map_first_name'] ) && isset( $fields[ $settings['field_map_first_name'] ] ) ) {
			$user_data['first_name'] = sanitize_text_field( $fields[ $settings['field_map_first_name'] ] );
		}
		
		if ( ! empty( $settings['field_map_last_name'] ) && isset( $fields[ $settings['field_map_last_name'] ] ) ) {
			$user_data['last_name'] = sanitize_text_field( $fields[ $settings['field_map_last_name'] ] );
		}
		
		$user_id = wp_insert_user( $user_data );

		if ( is_wp_error( $user_id ) ) {
			$ajax_handler->add_error_message( $user_id->get_error_message() );
			return;
		}

		// Handle Billing Phone (Always save if mapped, as it's used for login)
		$billing_phone_field = isset( $settings['field_map_billing_phone'] ) ? $settings['field_map_billing_phone'] : '';
		if ( ! empty( $billing_phone_field ) && isset( $fields[ $billing_phone_field ] ) ) {
			update_user_meta( $user_id, 'billing_phone', sanitize_text_field( $fields[ $billing_phone_field ] ) );
		}

		// Handle WooCommerce specific billing fields sync
		if ( class_exists( 'WooCommerce' ) ) {
			if ( ! empty( $user_data['first_name'] ) ) {
				update_user_meta( $user_id, 'billing_first_name', $user_data['first_name'] );
			}
			if ( ! empty( $user_data['last_name'] ) ) {
				update_user_meta( $user_id, 'billing_last_name', $user_data['last_name'] );
			}
		}

		// Email Verification
		$activation_key = md5( time() . $user_id );
		update_user_meta( $user_id, 'am_is_verified', '0' );
		update_user_meta( $user_id, 'am_activation_key', $activation_key );
		
		$args = [
			'am_action' => 'verify_email',
			'key'       => $activation_key,
			'uid'       => $user_id,
		];
		
		if ( ! empty( $settings['redirect_url'] ) ) {
			$args['redirect_to'] = urlencode( $settings['redirect_url'] );
		}
		
		$verification_link = add_query_arg( $args, home_url( '/' ) );
		
		$subject = esc_html__( 'Verify your email', 'user-registration-using-elementor-form' );
		$message = sprintf(
			/* translators: 1: Username 2: Verification Link */
			esc_html__( "Hi %1\$s,\n\nPlease verify your email by clicking the following link:\n%2\$s\n\nThanks!", 'user-registration-using-elementor-form' ),
			$username,
			$verification_link
		);
		
		wp_mail( $email, $subject, $message );
	}


	public function on_export( $element ) {
		return $element;
	}
	
	private function get_user_roles() {
		global $wp_roles;
		if ( ! isset( $wp_roles ) ) {
			return [];
		}
		
		return $wp_roles->get_names();
	}
}
